#include <LEDA/list.h>
#include <LEDA/graph.h>
#include <LEDA/graph_misc.h>
#include <LEDA/ugraph.h> 
#include <LEDA/templates/shortest_path.t>
#include "multilevel.h"
#include <math.h>
#include <time.h>

/////////////////////////////////////////////////////////////////////////////
Multilevel::Multilevel(CustomDesign *cd, float *args)
{
	CD = cd;
	levels.clear();
	//determine the levels
	int t = 1;
	while(1)
	{
		if((t * Pins_In_A_Tile) > CD->G.number_of_nodes())
			break;	//tiles is enough
		//split the quad
		t *= 4;
	}//end of determining the levels
	debug_print = 0;
	debug_count = 0;
	net_fail = 0;
	for(int i=0;i<CD->nets.size();i++)
		CD->nets[i].edge_routed = 0;

	multilevel_type = (int)args[0];
	alpha = args[1];

	if(multilevel_type == Routibility)
		DoMultilevel(t);
	else if(multilevel_type == PerformanceDriven)
		DoPerformanceMultilevel(t);
	Report();
}

/*
	dir = 0, this grid
	dir = +1, this grid's right
	dir = -1, this grid's left
	dir = +2, this grid's up
	dir = -2, this grid's down
	l means layer
*/
int Multilevel::GetDTGridValue(int t_index, int n, int m, int l, int dir)
{
	if(dir == 0)
		return levels[0].tiles[t_index].dt->layers[l].grids[n*dt_m + m];
	else if(dir == 1)
	{
		if(m+1<dt_m)
			return levels[0].tiles[t_index].dt->layers[l].grids[n*dt_m + m+1];
		else
		{
			if(t_index % levels[0].n == (levels[0].n-1))
				return 0;	//NSF
			else
				return levels[0].tiles[t_index+1].dt->layers[l].grids[n*dt_m + 0];
		}
	}
	else if(dir == -1)
	{
		if(m-1>=0)
			return levels[0].tiles[t_index].dt->layers[l].grids[n*dt_m + m-1];
		else
		{
			if(t_index % levels[0].n == 0)
				return 0;	//NSF
			else
				return levels[0].tiles[t_index-1].dt->layers[l].grids[n*dt_m + dt_m - 1];
		}
	}
	else if(dir == 2)
	{
		if(n+1<dt_n)
			return levels[0].tiles[t_index].dt->layers[l].grids[(n+1)*dt_m + m];
		else
		{
			if(t_index + levels[0].n > levels[0].n*levels[0].n)
				return 0;	//NSF
			else
				return levels[0].tiles[t_index+levels[0].n].dt->layers[l].grids[m];
		}
	}
	else if(dir == -2)
	{
		if(n-1>=0)
			return levels[0].tiles[t_index].dt->layers[l].grids[(n-1)*dt_m + m];
		else
		{
			if(t_index - levels[0].n < 0)
				return 0;	//NSF
			else
				return levels[0].tiles[t_index-levels[0].n].dt->layers[l].grids[(dt_n-1)*dt_m + m];
		}
	}
}

void Multilevel::DFSTimingAnalysis(CIP_ *cpi, double up_delay)
{
	double total_delay = up_delay;
	leda_list <leda_edge> to_deal_edges = CD->G.in_edges(cpi->n);
	list_item t = to_deal_edges.first();
	Edge_ *e = &CD->edges[CD->G.inf(to_deal_edges[t])];
	//more accurate calculation
	double resistance = 0;
	double this_wire_capacitance = 0;
	double this_wire_resistance = 0;

	if(e->routed)
	{
		resistance = (double)WireResistance * (double)e->dt_grids * (double)CD->wire_pitch / (double)scale_factor;
		this_wire_capacitance = ((double)WireCapacitance * (double)e->dt_grids * (double)CD->wire_pitch / (double)scale_factor);
		//consider gate
		total_delay += MinGateResistance * (this_wire_capacitance + cpi->d_c);
		//consider wire
		total_delay += this_wire_capacitance * (this_wire_capacitance/2 + cpi->d_c);
		//γ\ӦviainconnectiondelayA]ڭ̤wg[Fdelay
		total_delay += (double)e->vias * ViaDelay;
	}
	else
	{
		resistance = (double)WireResistance * (double)e->distance * (double)CD->wire_pitch / (double)scale_factor;
		this_wire_capacitance = ((double)WireCapacitance * (double)e->distance * (double)CD->wire_pitch / (double)scale_factor);
		//consider gate
		total_delay += MinGateResistance * (this_wire_capacitance + cpi->d_c);
		//consider wire
		total_delay += this_wire_capacitance * (this_wire_capacitance/2 + cpi->d_c);
		//n[2 * ViaDelay]OnetiWŵuAҥHĥworse case
		total_delay += 2 * ViaDelay;
	}
	cpi->delay = total_delay;

	to_deal_edges = CD->G.out_edges(cpi->n);
	while(to_deal_edges.size())
	{
		t = to_deal_edges.first();
		e = &CD->edges[CD->G.inf(to_deal_edges[t])];
		//source delay = 0
		DFSTimingAnalysis(e->target, total_delay);
		to_deal_edges.erase(t);
	}
}

double Multilevel::RoughDownStreamCapaciteance(CIP_ *cpi)
{
	double total_down_c = 0;
	
	leda_list <leda_edge> to_deal_edges = CD->G.out_edges(cpi->n);
	while(1)
	{
		list_item t = to_deal_edges.first();
		total_down_c += MinGateCapacitance;
		if(!to_deal_edges.size())
			break;
		Edge_ *e = &CD->edges[CD->G.inf(to_deal_edges[t])];

		if(e->routed)
			total_down_c += (double)WireCapacitance * (double)e->dt_grids * (double)CD->wire_pitch / (double)scale_factor;
		else
			total_down_c += (double)WireCapacitance * (double)e->distance * (double)CD->wire_pitch / (double)scale_factor;
		total_down_c += RoughDownStreamCapaciteance(e->target);
		to_deal_edges.erase(t);
	}
	CD->cellinstancepins[cpi->uid].d_c = total_down_c;
	return total_down_c;
}

/*
	input : a net, a sink
	use net topology to calculate Delay
	return Elmore delay of a sink from source (psec)
	Ҽ{dtedgeۥ
*/
void Multilevel::RoughAnalysisNetTreeDelay(int net_id)//(int net_id, double delay)
{
	CIP_ *cc = &CD->cellinstancepins[CD->nets[net_id].source];
	RoughDownStreamCapaciteance(cc);

	leda_list <leda_edge> to_deal_edges = CD->G.out_edges(cc->n);
	while(to_deal_edges.size())
	{
		list_item t = to_deal_edges.first();
		Edge_ *e = &CD->edges[CD->G.inf(to_deal_edges[t])];
		//source delay = 0
		DFSTimingAnalysis(e->target, 0);
		to_deal_edges.erase(t);
	}
}

void Multilevel::CalculateNetUpperAndLowerBound()
{

	for(int i=0;i<CD->nets.size();i++)
	{
		Net_ *n = &CD->nets[i];
		bool first = true;
		n->delay_upper_bound = 0;
		for(int j=0;j<n->pins_uid.size();j++)
		{
			if(n->pins_uid[j] == n->source)	continue;

			double delay = CD->cellinstancepins[n->pins_uid[j]].delay;
			if(first)
			{
				n->delay_upper_bound = delay;
				first = false;
			}
			else if(n->delay_upper_bound < delay)
				n->delay_upper_bound = delay;
		}
		n->max_delay = n->delay_upper_bound * alpha;
		constraint_net++;
		constraint_edges += n->pins_uid.size()-1;
		cout<<"n->delay_upper_bound:"<<n->delay_upper_bound<<endl;
		cout<<"n->max_delay:"<<n->max_delay<<endl;
	}
	CD->MinimumSpanningTreeRouting();
}

//clear the Multilevel instance
Multilevel::~Multilevel()
{
	for(int i=0;i<levels[0].tiles.size();i++)
	{
		if(levels[0].tiles[i].dt)
			delete levels[0].tiles[i].dt;
	}
}
//report tile information
void Multilevel::Report()
{
	cout << "\n##################  Multilevel Routing System Reports  ####################\n\n";
	int fail_edges = 0;
	int time_fail_edges = 0;
	int edges_never_meet = 0;
	for(int i=0;i<CD->edges.size();i++)
	{
		Edge_ *e = &CD->edges[i];
		CIP_ source, target;
		CD->GetEdgeST(e->e, source, target);
		if(!e->routed)
		{
			if(!fail_edges)
				cout << "The following nets fail to route:\n";
			cout<<"edge:"<<i<<"(net:"<<source.net_uid<<"( at sink:"<<target.name<<"("<<CD->cellinstances[target.ci_uid].name<<")"<<endl;
			fail_edges++;
		}
		else if(multilevel_type == PerformanceDriven)
		{
			if(!SatisfyRestrictedTimingConstraint(e->target))
				fail_edges++;
			else if(e->route_type == NerverMeetTiming)
				edges_never_meet++;
		}
	}
	time_fail_edges += edges_never_meet;
	cout << "Completed nets:"<<CD->edges.size()-fail_edges-edges_never_meet<<endl;
	cout << "total nets:"<<CD->edges.size()<<endl;
	cout << "Routing complete rating:"<< ((double)(CD->edges.size()-fail_edges-edges_never_meet)/(double)CD->edges.size())*100<<"%"<<endl<<endl;
}
/*
	return the index of tiles of level "l"
*/
int Multilevel::TileIndex(int ax, int ay, Level_ l)
{
	int x = (ax - CD->dmin_x)/l.tile_width;
	int y = (ay - CD->dmin_y)/l.tile_height;
	return l.n*y + x;
}
/*
	return the index of tiles of level 0
*/
int Multilevel::Level0TileIndex(CIP_ c)
{
	int x = (c.ax - CD->dmin_x)/levels[0].tile_width;
	int y = (c.ay - CD->dmin_y)/levels[0].tile_height;
	return levels[0].n*y + x;
}
/*
	^ǨIbLevel 0 tileWxindex
*/
void Multilevel::Level0TileXIndex(CIP_ source, CIP_ target, int &x1, int &x2)
{
	x1 = (source.ax - CD->dmin_x)/levels[0].tile_width;
	x2 = (target.ax - CD->dmin_x)/levels[0].tile_width;
}
/*
	^ǨIbLevel 0 tileWyۮtlƥ
*/
void Multilevel::Level0TileYIndex(CIP_ source, CIP_ target, int &y1, int &y2)
{
	y1 = (source.ay - CD->dmin_y)/levels[0].tile_height;
	y2 = (target.ay - CD->dmin_y)/levels[0].tile_height;
}
/*
	^ǨIbLevel 0 tileWxۮtlƥ
*/
int Multilevel::Level0TileXIndexDiff(CIP_ source, CIP_ target)
{
	int x1,x2;
	Level0TileXIndex(source, target, x1, x2);
	return abs(x1 - x2);
}
/*
	^ǨIbLevel 0 tileWyۮtlƥ
*/
int Multilevel::Level0TileYIndexDiff(CIP_ source, CIP_ target)
{
	int y1,y2;
	Level0TileYIndex(source, target, y1, y2);
	return abs(y1 - y2);
}
/*
	^ǨIbLevel 0 tileWxۮtlƥءAǤJOGm
*/
int Multilevel::Level0TileXIndexDiff2(int pos1, int pos2)
{
	int pos1_x = pos1 % levels[0].n;
	int pos2_x = pos2 % levels[0].n;
	return abs(pos1_x - pos2_x);
}
/*
	^ǨIbLevel 0 tileWyۮtlƥءAǤJOGm
*/
int Multilevel::Level0TileYIndexDiff2(int pos1, int pos2)
{
	int pos1_y = pos1 / levels[0].n;
	int pos2_y = pos2 / levels[0].n;
	return abs(pos1_y - pos2_y);
}
/*
	cost function
*/
double Multilevel::GetCost(Level_ *l, int index, int type)
{
	double unit = 0.333333;
	double a = unit, b = unit, c = unit;
	//parameters for cost function
	//]nnormalϥd 0 ~ 1
	double A = 1;
	int B = 2;
	int C,D;

	if(type)	//V
	{
		C = l->vcapacity;
		D = l->tiles[index].vdemand;
	}
	else	//H
	{
		C = l->hcapacity;
		D = l->tiles[index].hdemand;
	}
	
	DetailTile *dt = l->tiles[index].dt;
	return a * (A/(double)(pow(B,C-D))) + b * dt->overflow + c * 1;
}
/*
	Use Pattern L-Shape Routing to route edge
*/
void Multilevel::PatternRouting(Edge_ *e)
{
	Level_ *l = &levels[0];
	e->tile_routing.clear();
	CIP_ source, target;
	CD->GetEdgeST(e->e, source, target);
	int x1,x2,y1,y2;
	Level0TileXIndex(source, target, x1, x2);
	Level0TileYIndex(source, target, y1, y2);

	//horzontal routing
	if(y1==y2)
	{
		int min_x = min(x1,x2);
		int max_x = max(x1,x2);
		for(int i=min_x;i<max_x;i++)
		{
			int t_index = l->n*y1 + i;
			l->tiles[t_index].vdemand++;
			e->tile_routing.push_back(t_index);
		}
		int t_index = l->n*y1 + max_x;
		e->tile_routing.push_back(t_index);
		e->route_type = HLine;
		return;
	}
	//vertical routing
	if(x1==x2)
	{
		int min_y = min(y1,y2);
		int max_y = max(y1,y2);
		for(int i=min_y;i<max_y;i++)
		{
			int t_index = l->n*i + x1;
			l->tiles[t_index].hdemand++;
			e->tile_routing.push_back(t_index);
		}
		int t_index = l->n*max_y + x1;
		e->tile_routing.push_back(t_index);
		e->route_type = VLine;
		return;
	}
	/*
		Type 1:
		               |----*
		               *----|
		               x1   x2
		       routing_x1 = x1;
		       routing_x2 = x2;
		Type 2:
		               *----|
		               |----*
		               x1   x2
		       routing_x1 = x2;
		       routing_x2 = x1;
	*/
	int type;
	int routing_x1, routing_x2;
	int min_x, max_x, min_y, max_y;
	if(((x1-x2)*(y1-y2))>0)
	{
		type = 1;
		routing_x1 = min(x1, x2);
		routing_x2 = max(x1, x2);
	}
	else
	{
		type = 2;
		routing_x1 = max(x1, x2);
		routing_x2 = min(x1, x2);
	}
	min_x = min(x1, x2);
	max_x = max(x1, x2);
	min_y = min(y1, y2);
	max_y = max(y1, y2);
	//Top L routing
	double cost_top_l = 0;
	//routing_x1 bottom to routing_x1 top
	for(int i=min_y;i<max_y;i++)
	{
		int t_index = l->n*i + routing_x1;
		cost_top_l += GetCost(l, t_index, 0);
	}
	//left top to right top
	for(int i=min_x;i<max_x;i++)
	{
		int t_index = l->n*max_y + i;
		cost_top_l += GetCost(l, t_index, 1);
	}
	//Low L touting
	double cost_low_l = 0;
	//routing_x2 bottom to routing_x2 top
	for(int i=min_y;i<max_y;i++)
	{
		int t_index = l->n*i + routing_x2;
		cost_low_l += GetCost(l, t_index, 0);
	}
	//left bottom to right bottom
	for(int i=min_x;i<max_x;i++)
	{
		int t_index = l->n*min_y + i;
		cost_low_l += GetCost(l, t_index, 1);
	}
	//using Top L routing
	if(cost_low_l > cost_top_l)
	{
		for(int i=min_y;i<max_y;i++)
		{
			int t_index = l->n*i + routing_x1;
			l->tiles[t_index].hdemand++;
			if(i!=max_y)	//קK
				e->tile_routing.push_back(t_index);
		}
		for(int i=min_x;i<max_x;i++)
		{
			int t_index = l->n*max_y + i;
			l->tiles[t_index].vdemand++;
			e->tile_routing.push_back(t_index);
		}
		int t_index = l->n*max_y + max_x;
		e->tile_routing.push_back(t_index);
		e->route_type = TopL;
	}
	else
	{
		int t_index = l->n*max_y + routing_x2;
		e->tile_routing.push_back(t_index);
		for(int i=min_y;i<max_y;i++)
		{
			t_index = l->n*i + routing_x2;
			l->tiles[t_index].hdemand++;
			if(i!=min_y)	//קK
				e->tile_routing.push_back(t_index);
		}
		for(int i=min_x;i<max_x;i++)
		{
			t_index = l->n*min_y + i;
			l->tiles[t_index].vdemand++;
			e->tile_routing.push_back(t_index);
		}
		t_index = t_index = l->n*min_y + max_x;
		e->tile_routing.push_back(t_index);
		e->route_type = LowL;
	}
}
/*
	Use Pattern ZShape Routing to route edge
*/
void Multilevel::ZShapeRouting(Edge_ *e)
{
	TilePath_ tp;
	leda_list <int> path_index;
	vector <TilePath_> paths;
	path_index.clear();
	paths.clear();

	Level_ *l = &levels[0];
	CIP_ source, target;
	CD->GetEdgeST(e->e, source, target);
	int x1,x2,y1,y2;
	Level0TileXIndex(source, target, x1, x2);
	Level0TileYIndex(source, target, y1, y2);

	//horzontal routing
	if(y1==y2)
	{
		if(!net_fail++)
		{
			edges_failed_to_detail_route_at_coarsen.clear();
			cout<<"\n############## The following nets fail to route #####################\n";
		}
		edges_failed_to_detail_route_at_coarsen.push_back(e);
		cout<<" net:"<<e->source->net_uid;
		return;
	}
	//vertical routing
	if(x1==x2)
	{
		if(!net_fail++)
		{
			edges_failed_to_detail_route_at_coarsen.clear();
			cout<<"\n############## The following nets fail to route #####################\n";
		}
		edges_failed_to_detail_route_at_coarsen.push_back(e);
		cout<<" net:"<<e->source->net_uid;
		return;
	}
	/*
		Type 1:
		               |----*
		               *----|
		               x1   x2
		       routing_x1 = x1;
		       routing_x2 = x2;
		Type 2:
		               *----|
		               |----*
		               x1   x2
		       routing_x1 = x2;
		       routing_x2 = x1;
	*/
	int min_x, max_x, min_y, max_y;
	min_x = min(x1, x2);
	max_x = max(x1, x2);
	min_y = min(y1, y2);
	max_y = max(y1, y2);
	if(((x1-x2)*(y1-y2))>0)
	{
		//type = 1;
		//up and right and up
		for(int i=min_y+1;i<max_y;i++)
		{
			tp.tiles.clear();
			tp.tiles.push_back(l->n*min_y + min_x);
			tp.cost = 0;
			tp.cost += GetCost(l, l->n*min_y + min_x, 0);
			for(int j=min_y+1;j<=i;j++)
			{
				int t_index = l->n*j + min_x;
				if(j!=i)
					tp.cost += GetCost(l, t_index, 0);
				else
					tp.cost += GetCost(l, t_index, 1);
				tp.tiles.push_back(t_index);
			}
			for(int j=min_x+1;j<max_x;j++)
			{
				int t_index = l->n*i + j;
				tp.cost += GetCost(l, t_index, 1);
				tp.tiles.push_back(t_index);
			}
			for(int j=i;j<max_y;j++)
			{
				int t_index = l->n*j + max_x;
				tp.cost += GetCost(l, t_index, 0);
				tp.tiles.push_back(t_index);
			}
			tp.tiles.push_back(l->n*max_y + max_x);
			path_index.push_back(paths.size());
			paths.push_back(tp);
		}
		//right and up and right
		for(int i=min_x+1;i<max_x;i++)
		{
			tp.tiles.clear();
			tp.tiles.push_back(l->n*min_y + min_x);
			tp.cost = 0;
			tp.cost += GetCost(l, l->n*min_y + min_x, 1);
			for(int j=min_x+1;j<=i;j++)
			{
				int t_index = l->n*min_y + j;
				if(j!=i)
					tp.cost += GetCost(l, t_index, 1);
				else
					tp.cost += GetCost(l, t_index, 0);
				tp.tiles.push_back(t_index);
			}
			for(int j=min_y+1;j<max_y;j++)
			{
				int t_index = l->n*j + i;
				tp.cost += GetCost(l, t_index, 0);
				if(j!=min_y&&j!=max_y)
					tp.tiles.push_back(t_index);
			}
			for(int j=i;j<max_x;j++)
			{
				int t_index = l->n*max_y + j;
				tp.cost += GetCost(l, t_index, 1);
				tp.tiles.push_back(t_index);
			}
			tp.tiles.push_back(l->n*max_y + max_x);
			path_index.push_back(paths.size());
			paths.push_back(tp);
		}
	}
	else
	{
		//type = 2;
		//down and right and down
		for(int i=max_y-1;i>min_y;i--)
		{
			tp.tiles.clear();
			tp.tiles.push_back(l->n*max_y + min_x);
			tp.cost = 0;
			for(int j=max_y-1;j>=i;j--)
			{
				int t_index = l->n*j + min_x;
				tp.cost += GetCost(l, t_index, 0);
				if(j==i)
					tp.cost += GetCost(l, t_index, 1);
				tp.tiles.push_back(t_index);
			}
			for(int j=min_x+1;j<max_x;j++)
			{
				int t_index = l->n*i + j;
				tp.cost += GetCost(l, t_index, 1);
				tp.tiles.push_back(t_index);
			}
			for(int j=i;j>min_y;j--)
			{
				int t_index = l->n*j + max_x;
				if(j!=i)
					tp.cost += GetCost(l, t_index, 0);
				tp.tiles.push_back(t_index);
			}
			tp.cost += GetCost(l, l->n*min_y + max_x, 0);
			tp.tiles.push_back(l->n*min_y + max_x);
			path_index.push_back(paths.size());
			paths.push_back(tp);
		}
		//right and down and right
		for(int i=min_x+1;i<max_x;i++)
		{
			tp.tiles.clear();
			tp.tiles.push_back(l->n*max_y + min_x);
			tp.cost = 0;
			tp.cost += GetCost(l, l->n*max_y, 1);
			for(int j=min_x+1;j<=i;j++)
			{
				int t_index = l->n*max_y + j;
				if(j!=i)
					tp.cost += GetCost(l, t_index, 0);
				tp.tiles.push_back(t_index);
			}
			for(int j=max_y-1;j>min_y;j--)
			{
				int t_index = l->n*j + i;
				tp.cost += GetCost(l, t_index, 0);
				tp.tiles.push_back(t_index);
			}
			for(int j=i;j<max_x;j++)
			{
				int t_index = l->n*min_y + j;
				tp.cost += GetCost(l, t_index, 1);
				if(j==i)
					tp.cost += GetCost(l, t_index, 0);
				tp.tiles.push_back(t_index);
			}
			tp.tiles.push_back(l->n*min_y + max_x);
			path_index.push_back(paths.size());
			paths.push_back(tp);
		}
	}
	//Enumerate all Z-Shape Route
	while(path_index.size())
	{
		list_item min_item = path_index.first();
		list_item t_item = min_item;
		double min_cost = paths[path_index[min_item]].cost;
		while(1)
		{
			if(min_cost > paths[path_index[t_item]].cost)
			{
				min_item = t_item;
				min_cost = paths[path_index[t_item]].cost;
			}
			if(t_item == path_index.last())
				break;
			t_item = path_index.succ(t_item);
		}
		e->tile_routing.clear();
		for(int i=0;i<paths[path_index[min_item]].tiles.size();i++)
			e->tile_routing.push_back(paths[path_index[min_item]].tiles[i]);
		e->route_type = Maze;
		edges_to_detail_route.push_back(e);
		//detailed route successfully, exit(0)
		if(DetailRoute(1))
		{
			int before_t_index = paths[path_index[min_item]].tiles[0];
			for(int i=0;i<paths[path_index[min_item]].tiles.size();i++)
			{
				int t_index = paths[path_index[min_item]].tiles[i];
				//tile is neighbor in x
				if(Level0TileXIndexDiff2(before_t_index, t_index))
				{
					if(t_index < before_t_index)
						l->tiles[t_index].vdemand++;
					else
						l->tiles[before_t_index].vdemand++;
				}else if(Level0TileYIndexDiff2(before_t_index, t_index))//tile is neighbor in y
				{
					if(t_index < before_t_index)
						l->tiles[t_index].hdemand++;
					else
						l->tiles[before_t_index].hdemand++;
				}
				before_t_index = t_index;
			}
			return;
		}
		path_index.erase(min_item);
	}
	if(!net_fail++)
	{
		edges_failed_to_detail_route_at_coarsen.clear();
		cout<<"\n############## The following nets fail to route #####################\n";
	}
	edges_failed_to_detail_route_at_coarsen.push_back(e);
	cout<<" net:"<<e->source->net_uid;
}

/*
	Use Maze Routing to route edge
*/
void Multilevel::GlobalMazeRouting(Edge_ *e)
{
	TilePath_ tp;
	leda_list <int> path_index;
	vector <TilePath_> paths;
	//define a windows to perform GlobalMazeRouting
	vector <int> global_maze_window;
	path_index.clear();
	paths.clear();
	global_maze_window.clear();
	
	Level_ *l = &levels[0];
	CIP_ source, target;
	CD->GetEdgeST(e->e, source, target);
	int x1,x2,y1,y2;
	Level0TileXIndex(source, target, x1, x2);
	Level0TileYIndex(source, target, y1, y2);

	int min_x = min(x1 ,x2);
	int min_y = min(y1 ,y2);
	int max_x = max(x1 ,x2);
	int max_y = max(y1 ,y2);

	//define windows size by x1, x2, y1, y2
	double window_increase_ratio = 1.0;
	int min_window_width = 3;
	int min_window_height = 3;
	int window_width = abs(x1-x2) + 1;
	int window_height = abs(y1-y2) + 1;
	//more increase the search windows size also check if feasible
	int x_shift = (int)(window_increase_ratio * (double)window_width);
	if(x_shift<min_window_width)	x_shift = min_window_width;
	int y_shift = (int)(window_increase_ratio * (double)window_height);
	if(y_shift<min_window_height)	y_shift = min_window_height;
	int window_lx = min_x - x_shift;
	int window_rx = max_x + x_shift;
	int window_by = min_y - y_shift;
	int window_ty = max_y + y_shift;
	if(window_lx<0)	window_lx = 0;
	if(window_by<0)	window_by = 0;
	if(window_rx>=l->n)	window_rx = l->n-1;
	if(window_ty>=l->n)	window_ty = l->n-1;
	window_width = window_rx - window_lx + 1;
	window_height = window_ty - window_by + 1;

	//build Graph to perform search by Alg.
	GRAPH <int, double> MazeG;
	vector <leda_node> nodes;
	nodes.clear();

	MazeG.del_all_nodes();
	//assign node to present tiles
	for(int i=0;i<window_height;i++)
	{
		for(int j=0;j<window_width;j++)
		{
			int index = l->n*(window_by+i) + (window_lx+j);
			leda_node n;
			n = MazeG.new_node();
			MazeG.assign(n, index);
			nodes.push_back(n);
		}
	}

	leda_node s = nodes[window_width*(y1-window_by) + (x1-window_lx)];
	leda_node t = nodes[window_width*(y2-window_by) + (x2-window_lx)];

	int refinement_time = 20;
	int count = 0;
	double pre_cost = -100;
	int pre_cost_the_same;
	while(count++ < refinement_time)
	{
		//update cost of all edge every iteration
		MazeG.del_all_edges();
		//assign edge to cost passing a tile boundary
		for(int i=0;i<window_height;i++)
		{
			for(int j=0;j<window_width;j++)
			{
				double cost;
				int index = window_width*i + j;
				int t_index = l->n*(i+window_by) + (j+window_lx);
				leda_node n1, n2;
				int top_index, right_index;
				n1 = nodes[index];
				if(i!=window_height-1)
				{
					top_index = window_width*(i+1) + j;
					n2 = nodes[top_index];
					cost = GetCost(l, t_index, 0);
					//IInundirect graphbidirect graph~run shortest path Alog
					leda_edge e1 = MazeG.new_edge(n1, n2);
					MazeG.assign(e1, cost);
					e1 = MazeG.new_edge(n2, n1);
					MazeG.assign(e1, cost);
				}
				if(j!=window_width-1)
				{
					right_index = window_width*i + j + 1;
					n2 = nodes[right_index];
					cost = GetCost(l, t_index, 1);
					leda_edge e1 = MazeG.new_edge(n1, n2);
					MazeG.assign(e1, cost);
					e1 = MazeG.new_edge(n2, n1);
					MazeG.assign(e1, cost);
				}
			}
		}
		leda_node_array<double> dist(MazeG);	//C@node̵u|
		leda_node_array<leda_edge> pred(MazeG);
		leda_edge_array<double> cost(MazeG);

		leda_edge e1;
		forall_edges(e1,MazeG)
			cost[e1] = MazeG.inf(e1);

		SHORTEST_PATH_T(MazeG, s, cost, dist, pred);
		e->tile_routing.clear();

		if(pre_cost == dist[t])
		{
			if(pre_cost_the_same++>10)	//costs򳣤@ˡANi¶u\(pinmڥNQ]F)
			{
				cout<<"(*)";
				break;
			}
		}
		else
			pre_cost_the_same = 0;
		pre_cost = dist[t];
		e->tile_routing.push_back(MazeG.inf(t));
		leda_node n = t;
		while(1)
		{
			e1 = pred[n];
			if(e1==nil)	break;
			int source_index = MazeG.inf(MazeG.source(e1));

			e->tile_routing.push_back(source_index);
			n = MazeG.source(e1);
		}
		e->route_type = Maze;
		edges_to_detail_route.push_back(e);
		//detailed route successfully, exit(0)
		if(DetailRoute(0))
		{
			int before_t_index = e->tile_routing[0];
			for(int i=0;i<e->tile_routing.size();i++)
			{
				int t_index = e->tile_routing[i];
				//tile is neighbor in x
				if(Level0TileXIndexDiff2(before_t_index, t_index))
				{
					if(t_index < before_t_index)
						l->tiles[t_index].vdemand++;
					else
						l->tiles[before_t_index].vdemand++;
				}else if(Level0TileYIndexDiff2(before_t_index, t_index))//tile is neighbor in y
				{
					if(t_index < before_t_index)
						l->tiles[t_index].hdemand++;
					else
						l->tiles[before_t_index].hdemand++;
				}
				before_t_index = t_index;
			}
			return;
		}
	}

	e->tile_routing.clear();
	for(int i=window_by;i<=window_ty;i++)
	{
		for(int j=window_lx;j<=window_rx;j++)
		{
			int t_index = l->n*i + j;
			e->tile_routing.push_back(t_index);
		}
	}

	edges_to_detail_route.push_back(e);
	if(DetailRoute(0))
		return;

	if(!net_fail++)
	{
		cout<<"\n############## The following nets fail to route After Maze Route #####################\n";
		cout<<"############## (*) means net seems never be further refined      #####################\n";
	}
	cout<<"net:"<<e->source->net_uid<<" ";
}

void Multilevel::Coarsen(int t)
{
	leda_list <Edge_*> edge_list;
	edges_to_detail_route.clear();
	edge_list.clear();
	Edge_ *e;

	for(int i=0;i<CD->edges.size();i++)
	{
		e = &CD->edges[i];
		CIP_ source, target;
		CD->GetEdgeST(e->e, source, target);
		int source_index = TileIndex(source.ax, source.ay, levels[0]);
		//considering the local net to estimate routing resource
		if(source_index == TileIndex(target.ax, target.ay, levels[0]))
		{//local net
			e->level = 0;
			if(abs(source.ax-target.ax))
			{
				levels[0].tiles[source_index].vdemand++;
			}
			if(abs(source.ay-target.ay))
			{
				levels[0].tiles[source_index].hdemand++;
			}
			e->tile_routing.push_back(source_index);
			if(rand()%2)
				e->route_type = LowL;
			else
				e->route_type = TopL;
			edges_to_detail_route.push_back(e);
			if(!DetailRoute(1))
			{
				edge_list.push_back(e);
				cout<<"\n# net fail to route at level 0:"<<e->source->net_uid<<"#";
			}
		}
		else
			edge_list.push_back(e);
	}
	int n = levels[0].n;
	Level_ *pre_level = &levels[0];
	int lv = 1;	//current levels
	Level_ l;

	while(n!=1)
	{
		n/=2;
		ConstructLevel(pre_level, l, n);
		list_item traveler = edge_list.first();

		while(1)
		{
			CIP_ source, target;
			e = edge_list[traveler];
			CD->GetEdgeST(e->e, source, target);
			int source_index = TileIndex(source.ax, source.ay, l);
			if(source_index == TileIndex(target.ax, target.ay, l))
			{
				e->level = lv;
				PatternRouting(e);
				edges_to_detail_route.push_back(e);
				if(!DetailRoute(1))
					ZShapeRouting(e);
				if(traveler == edge_list.last())
				{
					edge_list.erase(traveler);
					break;
				}else
				{
					list_item traveler1 = traveler;
					traveler = edge_list.succ(traveler);
					edge_list.erase(traveler1);
					continue;
				}
			}
			if(traveler == edge_list.last())
				break;
			traveler = edge_list.succ(traveler);
		}
		levels.push_back(l);
		pre_level = &l;
		lv++;
	}
}

//chech if a local net satisfy timing constraint for "sink"
bool Multilevel::RelaxedSatisfyTimingConstraint(CIP_ *sink)
{
	double constraint = CD->nets[sink->net_uid].max_delay;
	if(constraint >= CD->NetElmoreDelay(sink->net_uid, sink->uid))
		return true;

	return false;
}
bool Multilevel::SatisfyRestrictedTimingConstraint(CIP_ *sink)
{
	double constraint = CD->nets[sink->net_uid].max_delay;
	if((constraint*control_ration) >= sink->delay)// CD->NetElmoreDelay(sink->net_uid, sink->uid))
		return true;

	return false;
}

//recalling modification, if succeeded, return true
bool Multilevel::RecallingModified(Edge_ *e)
{
	CIP_ *source = e->source;
	CIP_ *target = e->target;
	int edges_index = e->uid;
	//find a new parent meets timing constraint
	while(1)
	{
		leda_list <leda_edge> to_deal_edges = CD->G.in_edges(source->n);
		list_item t = to_deal_edges.first();

		//to the tree source and still violates timing constraint
		if(!to_deal_edges.size())
			return false;

		CIP_ new_source, tt;
		CD->GetEdgeST(to_deal_edges[t], new_source, tt);
		CD->UpdateEdge(edges_index, new_source.uid, target->uid);

		RoughAnalysisNetTreeDelay(source->net_uid);
		if(SatisfyRestrictedTimingConstraint(target))
			return true;

		//not meets timing, we continue search a new parent
		source = &new_source;
	}
}

void Multilevel::PerformanceCoarsen(int t)
{
	leda_list <Edge_*> edge_list;
	edges_to_detail_route.clear();
	edge_list.clear();
	Edge_ *e;
	for(int i=0;i<CD->edges.size();i++)
	{
		e = &CD->edges[i];
		CIP_ source, target;
		CD->GetEdgeST(e->e, source, target);
		int source_index = TileIndex(source.ax, source.ay, levels[0]);
		//considering the local net to estimate routing resource
		if(source_index == TileIndex(target.ax, target.ay, levels[0]))
		{//local net

			e->level = 0;
			if(abs(source.ax-target.ax))
			{
				levels[0].tiles[source_index].vdemand++;
			}
			if(abs(source.ay-target.ay))
			{
				levels[0].tiles[source_index].hdemand++;
			}
			if(rand()%2)
				e->route_type = LowL;
			else
				e->route_type = TopL;

			if(SatisfyRestrictedTimingConstraint(e->target))
			{
				e->tile_routing.push_back(source_index);
				edges_to_detail_route.push_back(e);
				if(!DetailRoute(1))
					edge_list.push_back(e);
			}else if(RecallingModified(e))
				i--;	//reprocess this Edge
			else
			{
				e->route_type = NerverMeetTiming;
			}
		}
		else
			edge_list.push_back(e);
	}
	int n = levels[0].n;
	Level_ *pre_level = &levels[0];
	int lv = 1;	//current levels
	Level_ l;

	while(n!=1)
	{
		n/=2;
		ConstructLevel(pre_level, l, n);
		list_item traveler = edge_list.first();
		while(1)
		{
			CIP_ source, target;
			e = edge_list[traveler];
			CD->GetEdgeST(e->e, source, target);
			int source_index = TileIndex(source.ax, source.ay, l);
			if(source_index == TileIndex(target.ax, target.ay, l))
			{
				e->level = lv;
				if(SatisfyRestrictedTimingConstraint(e->target))
				{
					while(1)
					{
						PatternRouting(e);
						edges_to_detail_route.push_back(e);
						if(!DetailRoute(1))
							ZShapeRouting(e);
							break;
					}
				}else if(RecallingModified(e))
					continue;	//reprocess this Edge
				else
				{
					e->route_type = NerverMeetTiming;
				}

				if(traveler == edge_list.last())
				{
					edge_list.erase(traveler);
					break;
				}else
				{
					list_item traveler1 = traveler;
					traveler = edge_list.succ(traveler);
					edge_list.erase(traveler1);
					continue;
				}
			}

			if(traveler == edge_list.last())
				break;
			traveler = edge_list.succ(traveler);
		}
		levels.push_back(l);
		pre_level = &l;
		lv++;
	}
}


void Multilevel::UnCoarsen()
{
	Edge_ *e;
	net_fail = 0;
	//refine edges_failed_to_detail_route_at_coarsen
	while(edges_failed_to_detail_route_at_coarsen.size())
	{
		//q̫@Ӷ}l
		list_item traveler = edges_failed_to_detail_route_at_coarsen.last();
		e = edges_failed_to_detail_route_at_coarsen[traveler];
		GlobalMazeRouting(e);
		edges_failed_to_detail_route_at_coarsen.erase(traveler);
	}
}


void Multilevel::Coarsen1(int t)
{
	//إߤ@link listAҦGedgeJAۦbC@levelɡAlocal net@ӭremove
	//AHαNtileresource¶local netNA@ҦedgeC
	leda_list <Edge_*> edge_list;
	edges_to_detail_route.clear();
	edge_list.clear();
	Edge_ *e;
	for(int i=0;i<CD->edges.size();i++)
	{
		e = &CD->edges[i];
		e->level = 0;
		CIP_ source, target;
		CD->GetEdgeST(e->e, source, target);
		int source_index = TileIndex(source.ax, source.ay, levels[0]);
		//considering the local net to estimate routing resource
		if(source_index == TileIndex(target.ax, target.ay, levels[0]))
		{//local net
			if(abs(source.ax-target.ax))
			{
				levels[0].tiles[source_index].vdemand++;
			}
			if(abs(source.ay-target.ay))
			{
				levels[0].tiles[source_index].hdemand++;
			}
			e->tile_routing.push_back(source_index);
			if(rand()%2)
				e->route_type = LowL;
			else
				e->route_type = TopL;
		}
		else
			PatternRouting(e);
		edges_to_detail_route.push_back(e);
		DetailRoute(1);
	}
}


void Multilevel::ConstructFirstLevel(int n)
{
	Level_ l;
	l.tiles.clear();
	int x_gap, y_gap;
	while(1)
	{
	l.n = n;
	x_gap = (int)((CD->dmax_x - CD->dmin_x)/n + 1);
	x_gap = ((x_gap-1) / CD->wire_pitch + 1) * CD->wire_pitch;
	y_gap = (int)((CD->dmax_y - CD->dmin_y)/n + 1);
	y_gap = ((y_gap-1) / CD->wire_pitch + 1) * CD->wire_pitch;
	dt_m = x_gap/CD->wire_pitch;
	dt_n = y_gap/CD->wire_pitch;
		if(dt_m > 60 || dt_n > 60)
			n*=2;
		else
			break;
	}

	l.hcapacity = CalculateHResource(x_gap);
	l.vcapacity = CalculateVResource(y_gap);
	for(int i=0;i<n;i++)
		for(int j=0;j<n;j++)
		{
			Tile_ t;
			t.lx = CD->dmin_x + x_gap * j;
			t.rx = t.lx + x_gap;
			t.by = CD->dmin_y + y_gap * i;
			t.ty = t.by + y_gap;
			t.hdemand = 0;
			t.vdemand = 0;
			t.dt = NULL;
			t.dt = new DetailTile(dt_m, dt_n, CD->number_of_layers);
			l.tiles.push_back(t);
		}
	l.tile_width = x_gap;
	l.tile_height = y_gap;
	levels.push_back(l);
}

int Multilevel::CalculateResourceFromPreLevel(Level_ *pre_level, int i, int j, int type)
{
	int n = pre_level->n;
	if(!type)	//hdemand
		return (pre_level->tiles[2*n*i+j].hdemand+pre_level->tiles[2*n*i+j+1].hdemand+pre_level->tiles[2*n*i+n+j+1].hdemand+pre_level->tiles[2*n*i+n+j+1].hdemand);
	return (pre_level->tiles[2*n*i+j].vdemand+pre_level->tiles[2*n*i+j+1].vdemand+pre_level->tiles[2*n*i+n+j+1].vdemand+pre_level->tiles[2*n*i+n+j+1].vdemand);
}

void Multilevel::ConstructLevel(Level_ *pre_level, Level_ &l, int n)
{
	l.tiles.clear();
	l.n = n;
	l.vcapacity = 4*pre_level->vcapacity;
	l.hcapacity = 4*pre_level->hcapacity;
	l.tile_width = 2*pre_level->tile_width;
	l.tile_height = 2*pre_level->tile_height;
	int x_gap = (int)((CD->dmax_x - CD->dmin_x)/n);
	int y_gap = (int)((CD->dmax_y - CD->dmin_y)/n);
	for(int i=0;i<n;i++)
		for(int j=0;j<n;j++)
		{
			Tile_ t;
			t.lx = CD->dmin_x + x_gap * j;
			t.rx = t.lx + x_gap;
			t.by = CD->dmin_y + y_gap * i;
			t.ty = t.by + y_gap;
			t.hdemand = CalculateResourceFromPreLevel(pre_level, i, j, 0);
			t.vdemand = CalculateResourceFromPreLevel(pre_level, i, j, 1);
			l.tiles.push_back(t);
		}
}

int Multilevel::CalculateHResource(int x_gap)
{
	//hroizontal resource = width/min(wire width+spacing) * layers
	return ((int)(x_gap/CD->wire_pitch)*CD->number_of_layers);
}
int Multilevel::CalculateVResource(int y_gap)
{
	//vertical resource = height/min(wire width+spacing) * layers
	return ((int)(y_gap/CD->wire_pitch)*CD->number_of_layers);
}

/*
	Maze Searching Alg
*/
bool Multilevel::DetailedMazeSearching(int &s_x, int &s_y, int t_x, int t_y, int mark, int Obstacle, int &fail_pos)
{

//*from target, search 4 way - Up(search_way=1), Right(search_way=2)
//, Down(search_way=3) and Left(search_way=4).
leda_list<int> list_last_way;
leda_list<int> grid_list_x;
leda_list<int> grid_list_y;
leda_list<int> list_cost;

	grid_list_x.clear();
	grid_list_y.clear();
	list_cost.clear();
	list_last_way.clear();
	grid_list_x.push_back(t_x);
	grid_list_y.push_back(t_y);
	list_cost.push_back(0);
	list_last_way.push_back(0);
	
	int x, y;
	while(grid_list_x.size())
	{
		x = grid_list_x[grid_list_x.first()];
		y = grid_list_y[grid_list_y.first()];
		int cost = list_cost[list_cost.first()]+1;
		int last_way = list_last_way[list_last_way.first()];
		//ܤ֭n@ӤVsearch\A_ho@gridN]w = Obstacle
		int last_one_dir_succ = 0;
		int original_grid = y * maze_width + x;
		int set_original_h = 0;
		int set_original_v = 0;
		//*from source, search 4 way - Up(search_way=1), Right(search_way=2)
		//, Down(search_way=3) and Left(search_way=4).
		//**search Up
		int area_size = h_maze_area.size();
		int search = (y+1) * maze_width + x;
		//path is going to be found
		if((area_size > search) && (h_maze_area[search] == CD->NetPin || v_maze_area[search] == CD->NetPin))
		{
			if(last_way != 1 && last_way != 0)
			{
				if(v_maze_area[original_grid] == 0)
				{
					v_maze_area[original_grid] = cost - 1;
					v_maze_area[search] = mark;
					h_maze_area[search] = mark;
					s_x = x;
					s_y = y+1;
					return true;
				}
			}
			else
			{
				v_maze_area[search] = mark;
				h_maze_area[search] = mark;
				s_x = x;
				s_y = y+1;
				return true;
			}
		}
		if((area_size > search)&&(!v_maze_area[search]))	//search is ok!
		{
			bool fail = 0;
			if(last_way != 1 && last_way != 0)
			{
				if(v_maze_area[original_grid] < 0)
				{
					fail = true;
				}
				else
					set_original_v = 1;
			}
			if(!fail)
			{
				v_maze_area[search] = cost;
				grid_list_x.push_back(x);
				grid_list_y.push_back(y+1);
				list_cost.push_back(cost);
				list_last_way.push_back(1);
				last_one_dir_succ = 1;
			}
		}
		//**search Right
		search = y * maze_width + x + 1;
		//path is going to be found
		if(((x+1) < maze_width)&&(h_maze_area[search] == CD->NetPin || v_maze_area[search] == CD->NetPin))
		{
			if(last_way != 2 && last_way != 0)
			{
				if(h_maze_area[original_grid] == 0)
				{
					h_maze_area[original_grid] = cost - 1;
					v_maze_area[search] = mark;
					h_maze_area[search] = mark;
					s_x = x+1;
					s_y = y;
					return true;
				}
			}
			else
			{
				v_maze_area[search] = mark;
				h_maze_area[search] = mark;
				s_x = x+1;
				s_y = y;
				return true;
			}
		}
		if(((x+1) < maze_width)&&(!h_maze_area[search]))	//search is ok!
		{
			bool fail = 0;
			if(last_way != 2 && last_way != 0)
			{
				if(h_maze_area[original_grid] < 0)
				{
					fail = true;
				}
				else
					set_original_h = 1;
			}
			if(!fail)
			{
				h_maze_area[search] = cost;
				grid_list_x.push_back(x+1);
				grid_list_y.push_back(y);
				list_cost.push_back(cost);
				list_last_way.push_back(2);
				last_one_dir_succ = 1;
			}
		}
		//**search Down
		search = (y-1) * maze_width + x;
		//path is going to be found
		if(((y-1) >= 0)&&(h_maze_area[search] == CD->NetPin || v_maze_area[search] == CD->NetPin))
		{
			if(last_way != 3 && last_way != 0)
			{
				if(v_maze_area[original_grid] == 0)
				{
					v_maze_area[original_grid] = cost - 1;
					v_maze_area[search] = mark;
					h_maze_area[search] = mark;
					s_x = x;
					s_y = y-1;
					return true;
				}
			}
			else
			{
				v_maze_area[search] = mark;
				h_maze_area[search] = mark;
				s_x = x;
				s_y = y-1;
				return true;
			}
		}
		if(((y-1) >= 0)&&(!v_maze_area[search]))	//search is ok!
		{
			bool fail = 0;
			if(last_way != 3 && last_way != 0)
			{
				if(v_maze_area[original_grid] < 0)
				{
					fail = true;
				}
				else
					set_original_v = 1;
			}
			if(!fail)
			{
				v_maze_area[search] = cost;
				grid_list_x.push_back(x);
				grid_list_y.push_back(y-1);
				list_cost.push_back(cost);
				list_last_way.push_back(3);
				last_one_dir_succ = 1;
			}
		}
		//**search Left
		search = y * maze_width + x - 1;
		//path is going to be found
		if(((x-1) >= 0)&&(h_maze_area[search] == CD->NetPin || v_maze_area[search] == CD->NetPin))
		{
			if(last_way != 4 && last_way != 0)
			{
				if(h_maze_area[original_grid] == 0)
				{
					h_maze_area[original_grid] = cost - 1;
					v_maze_area[search] = mark;
					h_maze_area[search] = mark;
					s_x = x-1;
					s_y = y;
					return true;
				}
			}
			else
			{
				v_maze_area[search] = mark;
				h_maze_area[search] = mark;
				s_x = x-1;
				s_y = y;
				return true;
			}
		}
		if(((x-1) >= 0)&&(!h_maze_area[search]))	//search is ok!
		{
			bool fail = 0;
			if(last_way != 4 && last_way != 0)
			{
				if(h_maze_area[original_grid] < 0)
				{
					fail = true;
				}
				else
					set_original_h = 1;
			}
			if(!fail)
			{
				h_maze_area[search] = cost;
				grid_list_x.push_back(x-1);
				grid_list_y.push_back(y);
				list_cost.push_back(cost);
				list_last_way.push_back(4);
				last_one_dir_succ = 1;
			}
		}
		if(set_original_h)
			h_maze_area[original_grid] = cost-1;
		if(set_original_v)
			v_maze_area[original_grid] = cost-1;
		if(!last_one_dir_succ && ((v_maze_area[original_grid] < 0)||(h_maze_area[original_grid] < 0)))
			h_maze_area[original_grid] = v_maze_area[original_grid] = Obstacle;
		grid_list_x.erase(grid_list_x.first());
		grid_list_y.erase(grid_list_y.first());
		list_cost.erase(list_cost.first());
		list_last_way.erase(list_last_way.first());
	}

	fail_pos = y * maze_width + x;
	return false;
}
/*
	After Maze Searching Alg. mark all grid with distance from source to target,
	we backtrace to find a real desire path.
	(Using h_maze_area, v_maze_area and marking grids with "mark".)
	route_type is use to guide best route
*/
void Multilevel::BacktraceToFindPath(int s_x, int s_y, int t_x, int t_y, int mark, int route_type, int &vias)
{
	int min_grids = INT_MAX;
	//*from target, search 4 way - Up(search_way=1), Right(search_way=2)
	//, Down(search_way=3) and Left(search_way=4).
	//**search Up(We except it will use v_maze_area now!)
	int area_size = h_maze_area.size();
	int search = (s_y+1) * maze_width + s_x;
	if(area_size > search)	//search is ok!
		if(v_maze_area[search]>0 && (v_maze_area[search]<min_grids))
			min_grids = v_maze_area[search];
	search = s_y * maze_width + s_x + 1;
	//**search Right(We except it will use h_maze_area now!)
	if((s_x+1) < maze_width)	//search is ok!
		if(h_maze_area[search]>0 && (h_maze_area[search]<min_grids))
			min_grids = h_maze_area[search];
	search = (s_y-1) * maze_width + s_x;
	//**search Down(We except it will use v_maze_area now!)
	if((s_y-1) >= 0)	//search is ok!
		if(v_maze_area[search]>0 && (v_maze_area[search]<min_grids))
			min_grids = v_maze_area[search];
	search = s_y * maze_width + s_x - 1;
	//**search Left(We except it will use h_maze_area now!)
	if((s_x-1) >= 0)	//search is ok!
		if(h_maze_area[search]>0 && (h_maze_area[search]<min_grids))
			min_grids = h_maze_area[search];
	
	int last_way = 0;
	int x = s_x;
	int y = s_y;
	int cost = min_grids + 1;
	int suggest_route = 0;
	int HV_H = 0;
	int HV_V = 1;
	int HV_ANY = 2;
	if(route_type == TopL)
	{
		if(t_y >= s_y)
			suggest_route = HV_V;	//suggest for Top
		else
			suggest_route = HV_H;	//suggest for Left, Right
	}
	else
	{
		if(t_y <= s_y)
			suggest_route = HV_V;	//suggest for Down
		else
			suggest_route = HV_H;	//suggest for Left, Right
	}

int cc = 0;
	vias = 0;
	while(1)
	{
		int now_x;
		int now_y;
		int now_search, oringinal_search;
		oringinal_search = y * maze_width + x;

		//*from target, search 4 way - Up(search_way=1), Right(search_way=2)
		//, Down(search_way=3) and Left(search_way=4).
		//**search Up(We except it will use v_maze_area now!)

		//initial for up search
		int area_size = h_maze_area.size();
		now_x = x;
		now_y = y+1;
		now_search = now_y * maze_width + now_x;
		//path is found
		if(now_search == (t_y * maze_width + t_x))
		{
			if(last_way != 1 && last_way != 0)
				vias++;
			return;
		}
		if(suggest_route == HV_V || suggest_route == HV_ANY)
		{
			suggest_route = HV_ANY;
			while(area_size > now_search && (v_maze_area[now_search]) == cost-1 && (cost-1))
			{
				v_maze_area[now_search] = mark;
				if(last_way != 1 && last_way != 0)
				{
					
					vias++;
					v_maze_area[oringinal_search] = mark;
				}
				oringinal_search = now_y * maze_width + now_x;
				y = now_y;
				cost = cost-1;
				now_y = now_y+1;
				now_search = now_y * maze_width + now_x;
				//path is found
				if(now_search == (s_y * maze_width + s_x))
					return;
				last_way = 1;
			}
		}
		//path is found
		if(now_search == (t_y * maze_width + t_x))
			return;
		if(last_way == 1)
			h_maze_area[oringinal_search] = mark;
		//initial for down search
		now_x = x;
		now_y = y - 1;
		now_search = now_y * maze_width + now_x;
		//path is found
		if(now_search == (t_y * maze_width + t_x))
		{
			if(last_way != 3 && last_way != 0)
				vias++;
			return;
		}
		if(suggest_route == HV_V || suggest_route == HV_ANY)
		{
			suggest_route = HV_ANY;
			while(now_y >= 0 && (v_maze_area[now_search]) == cost-1 && (cost-1))
			{
				v_maze_area[now_search] = mark;
				if(last_way != 3 && last_way != 0)
				{
					vias++;
					v_maze_area[oringinal_search] = mark;
				}
				oringinal_search = now_y * maze_width + now_x;
				y = now_y;
				cost = cost-1;
				now_y = now_y-1;
				now_search = now_y * maze_width + now_x;
				//path is found
				if(now_search == (s_y * maze_width + s_x))
					return;
				last_way = 3;
			}
		}
		//path is found
		if(now_search == (t_y * maze_width + t_x))
			return;
		if(last_way == 3)
			h_maze_area[oringinal_search] = mark;
		//initial for right search
		now_x = x + 1;
		now_y = y;
		now_search = now_y * maze_width + now_x;
		//path is found
		if(now_search == (t_y * maze_width + t_x))
		{
			if(last_way != 2 && last_way != 0)
				vias++;
			return;
		}
		if(suggest_route == HV_H || suggest_route == HV_ANY)
		{
			suggest_route = HV_ANY;
			while((now_x) < maze_width && (h_maze_area[now_search]) == cost-1 && (cost-1))
			{
				h_maze_area[now_search] = mark;
				if(last_way != 2 && last_way != 0)
				{
					vias++;
					h_maze_area[oringinal_search] = mark;
				}
				oringinal_search = now_y * maze_width + now_x;
				x = now_x;
				cost = cost-1;
				now_x = now_x+1;
				now_search = now_y * maze_width + now_x;
				//path is found
				if(now_search == (s_y * maze_width + s_x))
					return;
				last_way = 2;
			}
		}
		//path is found
		if(now_search == (t_y * maze_width + t_x))
			return;
		if(last_way == 2)
			v_maze_area[oringinal_search] = mark;
		//initial for left search
		now_x = x - 1;
		now_y = y;
		now_search = now_y * maze_width + now_x;
		//path is found
		if(now_search == (t_y * maze_width + t_x))
		{
			if(last_way != 4 && last_way != 0)
				vias++;
			return;
		}
		if(suggest_route == HV_H || suggest_route == HV_ANY)
		{
			suggest_route = HV_ANY;
			while(now_x >= 0 && (h_maze_area[now_search]) == cost-1 && (cost-1))
			{
				h_maze_area[now_search] = mark;
				if(last_way != 4 && last_way != 0)
				{
					vias++;
					h_maze_area[oringinal_search] = mark;
				}
				oringinal_search = now_y * maze_width + now_x;
				x = now_x;
				cost = cost-1;
				now_x = now_x-1;
				now_search = now_y * maze_width + now_x;
				//path is found
				if(now_search == (s_y * maze_width + s_x))
					return;
				last_way = 4;
			}
		}
		//path is found
		if(now_search == (t_y * maze_width + t_x))
			return;
		if(last_way == 4)
			v_maze_area[oringinal_search] = mark;
		if(cc++==1000)
		{
			cout<<"mark:"<<mark<<" ERR\n";


			cout<<"#######\nv maze:"<<endl;
			for(int k=0;k<maze_width*maze_height;k++)
			{
				printf(" %6d",v_maze_area[k]);
				if(k%maze_width==(maze_width-1))
					cout<<endl;
			}
			cout<<"\n#######\nh maze:"<<endl;
			for(int k=0;k<maze_width*maze_height;k++)
			{
				printf(" %6d",h_maze_area[k]);
				if(k%maze_width==(maze_width-1))
					cout<<endl;
			}

			exit(0);
		}
	}
}

/*
	ǤJdt->grids(t)AL̬OOݩP@net
*/
bool Multilevel::TheSameNet(int v1, int v2)
{
	if(!v1||!v2)
		return false;
	if(((-1)*v1 > CD->edges.size()) || ((-1)*v2 > CD->edges.size()))
		return false;
	Edge_ *e1 = &CD->edges[(-1)*v1-1];
	Edge_ *e2 = &CD->edges[(-1)*v2-1];
	if(e1->source->net_uid == e2->source->net_uid)
		return true;
	else
		return false;
}


/*
	According to the tile assignment of global route, we detail route 
	all edges of "edges_to_detail_route", each tile by grid maze route.
	We perform the maze route and layer assignment sequentially.
		
	*ΦҼ{obstacle(gate)hmA]ڭ̱ĥover-the-cell routing
	pGnsource HtargetΦrectangle boxӰh_maze_area
	type = 1
	MNObGlobalMazeRouting Global RouteLϰөwq
*/
bool Multilevel::DetailRoute(int type)
{

int start_layer_type = HV_H;
int up = 1, down = -1;
int this_layer_with_up_layer_or_down;
int where_is_start_layer;
int i=0;
	//perform the maze route and layer assignment sequentially.

	while(edges_to_detail_route.size())
	{
		Edge_ *e = edges_to_detail_route[edges_to_detail_route.first()];
		int mark = -1*(e->uid)-1;

		CIP_ source, target;
		CD->GetEdgeST(e->e, source, target);
		int s_x, t_x, s_y, t_y;
		Level0TileXIndex(source, target, s_x, t_x);
		Level0TileYIndex(source, target, s_y, t_y);
		int x_diff = abs(s_x - t_x);
		int y_diff = abs(s_y - t_y);

		h_maze_area.clear();
		v_maze_area.clear();

		int min_x_index = min(s_x, t_x);
		int min_y_index = min(s_y, t_y);
		int left_bottom_corner_index;
		//Setup Detailed Maze Area
		if(type == 1)
		{
			maze_width = dt_m * (x_diff + 1);
			maze_height = dt_n * (y_diff + 1);
		}else
		{
			int min_x, min_y, max_x, max_y;
			int x1, y1;
			Level_ *l = &levels[0];
				
			for(int i=0;i<e->tile_routing.size();i++)
			{
				int t_index = e->tile_routing[i];
				x1 = t_index % l->n;
				y1 = t_index / l->n;
				if(!i)
				{
					min_x = x1;
					min_y = y1;
					max_x = x1;
					max_y = y1;
				}else
				{
					if(x1 < min_x)	min_x = x1;
					if(y1 < min_y)	min_y = y1;
					if(x1 > max_x)	max_x = x1;
					if(y1 > max_y)	max_y = y1;
				}
			}
			maze_width = dt_m * (max_x-min_x + 1);
			maze_height = dt_n * (max_y-min_y + 1);
			min_x_index = min_x;
			min_y_index = min_y;
		}
		h_maze_area.resize(maze_width*maze_height);
		v_maze_area.resize(maze_width*maze_height);
		left_bottom_corner_index = levels[0].n * min_y_index + min_x_index;

		DetailTile *dt = levels[0].tiles[0].dt;
		//we first try route each layer until succeed,
		//if all layers fail, return fail message.
		bool first = true;
		bool last_try = false;
		if(CD->number_of_layers < 2)	{cout<<"Not support 1 layer route\n"; exit(0);}
		while(1)	//try all layers
		{
			if(first)
			{
				if(start_layer_type == dt->layers[0].type)
				{
					this_layer_with_up_layer_or_down = up;
					where_is_start_layer = 0;
				}
				else
				{
					this_layer_with_up_layer_or_down = down;
					where_is_start_layer = 1;
				}
				first = false;
			}
			else
			{
				if(this_layer_with_up_layer_or_down == down)
				{//try with up
					if(CD->number_of_layers > where_is_start_layer + 1)
						this_layer_with_up_layer_or_down = up;
				}
				else
				{
					if(CD->number_of_layers > where_is_start_layer + 2)
					{
						this_layer_with_up_layer_or_down = down;
						where_is_start_layer += 2;
					}else
						break;	//never happen
				}
			}
			if(((where_is_start_layer+1) == CD->number_of_layers) || (((where_is_start_layer+2) == CD->number_of_layers)&&this_layer_with_up_layer_or_down == up))
				last_try = true;

			//initial the maze_area, set occupied
			//(later unoccupied grid will be recovered).
			for(int k=0;k<h_maze_area.size();k++)
			{
				h_maze_area[k] = CD->PinObstacle;
				v_maze_area[k] = CD->PinObstacle;
			}
			//recover unoccupied grid
			for(int k=0;k<e->tile_routing.size();k++)
			{
				int x_diff2 = Level0TileXIndexDiff2(left_bottom_corner_index, e->tile_routing[k]);
				int y_diff2 = Level0TileYIndexDiff2(left_bottom_corner_index, e->tile_routing[k]);
				int maze_area_y_add = maze_width*dt_n*y_diff2;

				int set_for_the_same_net;
				//(1)routibility
				if(multilevel_type == Routibility)
					set_for_the_same_net = 0;
				//(2)Performance
				else if(multilevel_type == PerformanceDriven)
					set_for_the_same_net = 0;//CD->PinObstacle;

				dt = levels[0].tiles[e->tile_routing[k]].dt;
				for(int l=0;l<dt->layers[0].grids.size();l++)
				{
					int maze_area_x_add = dt_m*x_diff2 + (l/dt_m)*maze_width;
					int maze_area_add = maze_area_x_add+maze_area_y_add;

					if(start_layer_type == HV_V)
					{
						if(TheSameNet(dt->layers[where_is_start_layer].grids[l], mark))
							v_maze_area[maze_area_add + l%dt_m] = set_for_the_same_net;
						else
							v_maze_area[maze_area_add + l%dt_m] = dt->layers[where_is_start_layer].grids[l];
						if(TheSameNet(dt->layers[where_is_start_layer+this_layer_with_up_layer_or_down].grids[l], mark))
							h_maze_area[maze_area_add + l%dt_m] = set_for_the_same_net;
						else
							h_maze_area[maze_area_add + l%dt_m] = dt->layers[where_is_start_layer+this_layer_with_up_layer_or_down].grids[l];
					}else
					{
						if(TheSameNet(dt->layers[where_is_start_layer].grids[l], mark))
							h_maze_area[maze_area_add + l%dt_m] = set_for_the_same_net;
						else
							h_maze_area[maze_area_add + l%dt_m] = dt->layers[where_is_start_layer].grids[l];
						if(TheSameNet(dt->layers[where_is_start_layer+this_layer_with_up_layer_or_down].grids[l], mark))
							v_maze_area[maze_area_add + l%dt_m] = set_for_the_same_net;
						else
							v_maze_area[maze_area_add + l%dt_m] = dt->layers[where_is_start_layer+this_layer_with_up_layer_or_down].grids[l];
					}
				}
			}

			//*now, maze search starts. We start at Maze Area and then mark back to the
			// corresponding grids in a detail grids.

			//**first,find the source and target position on the detail grids(Maze Area)
			//***source
			int x_diff2 = s_x - min_x_index;
			int y_diff2 = s_y - min_y_index;
			//***target
			//]source and targetObY@Tile̥UAҥHnҼ{Ut
			int maze_area_x_in_detailgrid = ((source.ax - CD->dmin_x) % levels[0].tile_width)/CD->wire_pitch;
			int maze_area_y_in_detailgrid = ((source.ay - CD->dmin_y) % levels[0].tile_height)/CD->wire_pitch;
			
			int s_maze_area_x = dt_m*x_diff2 + maze_area_x_in_detailgrid;
			int s_maze_area_y = dt_n*y_diff2 + maze_area_y_in_detailgrid;
			//***target

			int s_index = s_maze_area_y * maze_width + s_maze_area_x;
			h_maze_area[s_index] = v_maze_area[s_index] = CD->NetPin;//mark

			//ڭ̶}pinkI¶uAHWiroutibility(M]iH}WUAOnݱp)
			if(s_maze_area_x+1 < maze_width)
			{
				s_index = s_maze_area_y * maze_width + s_maze_area_x+1;
				if(TheSameNet(h_maze_area[s_index], mark))
					h_maze_area[s_index] = 0;
				if(TheSameNet(v_maze_area[s_index], mark))
					v_maze_area[s_index] = 0;
			}
			if(s_maze_area_x+2 < maze_width)
			{
				s_index = s_maze_area_y * maze_width + s_maze_area_x+2;
				if(TheSameNet(h_maze_area[s_index], mark))
					h_maze_area[s_index] = 0;
				if(TheSameNet(v_maze_area[s_index], mark))
					v_maze_area[s_index] = 0;
			}

			if(s_maze_area_x-1 >= 0)
			{
				s_index = s_maze_area_y * maze_width + s_maze_area_x-1;
				if(TheSameNet(h_maze_area[s_index], mark))
					h_maze_area[s_index] = 0;
				if(TheSameNet(v_maze_area[s_index], mark))
					v_maze_area[s_index] = 0;
			}
			if(s_maze_area_x-2 >= 0)
			{
				s_index = s_maze_area_y * maze_width + s_maze_area_x-2;
				if(TheSameNet(h_maze_area[s_index], mark))
					h_maze_area[s_index] = 0;
				if(TheSameNet(v_maze_area[s_index], mark))
					v_maze_area[s_index] = 0;
			}

			if(((s_maze_area_y+1) * maze_width + s_maze_area_x)<h_maze_area.size())
			{
				s_index = (s_maze_area_y+1) * maze_width + s_maze_area_x;
				if(TheSameNet(h_maze_area[s_index], mark))
					h_maze_area[s_index] = 0;
				if(TheSameNet(v_maze_area[s_index], mark))
					v_maze_area[s_index] = 0;
			}
			if(((s_maze_area_y+2) * maze_width + s_maze_area_x)<h_maze_area.size())
			{
				s_index = (s_maze_area_y+2) * maze_width + s_maze_area_x;
				if(TheSameNet(h_maze_area[s_index], mark))
					h_maze_area[s_index] = 0;
				if(TheSameNet(v_maze_area[s_index], mark))
					v_maze_area[s_index] = 0;
			}
			if(s_maze_area_y-1 >= 0)
			{
				s_index = (s_maze_area_y-1) * maze_width + s_maze_area_x;
				if(TheSameNet(h_maze_area[s_index], mark))
					h_maze_area[s_index] = 0;
				if(TheSameNet(v_maze_area[s_index], mark))
					v_maze_area[s_index] = 0;
			}
			if(s_maze_area_y-2 >= 0)
			{
				s_index = (s_maze_area_y-2) * maze_width + s_maze_area_x;
				if(TheSameNet(h_maze_area[s_index], mark))
					h_maze_area[s_index] = 0;
				if(TheSameNet(v_maze_area[s_index], mark))
					v_maze_area[s_index] = 0;
			}
			if((((s_maze_area_y+1) * maze_width + s_maze_area_x)<h_maze_area.size())&&
			  (s_maze_area_x-1 >= 0))
			{
				s_index = (s_maze_area_y+1) * maze_width + s_maze_area_x-1;
				if(TheSameNet(h_maze_area[s_index], mark))
					h_maze_area[s_index] = 0;
				if(TheSameNet(v_maze_area[s_index], mark))
					v_maze_area[s_index] = 0;
			}
			if((((s_maze_area_y+1) * maze_width + s_maze_area_x)<h_maze_area.size())&&
			  (s_maze_area_x+1 < maze_width))
			{
				s_index = (s_maze_area_y+1) * maze_width + s_maze_area_x+1;
				if(TheSameNet(h_maze_area[s_index], mark))
					h_maze_area[s_index] = 0;
				if(TheSameNet(v_maze_area[s_index], mark))
					v_maze_area[s_index] = 0;
			}
			if(((s_maze_area_x-1 >= 0))&&
			  (s_maze_area_x-1 >= 0))
			{
				s_index = (s_maze_area_y-1) * maze_width + s_maze_area_x-1;
				if(TheSameNet(h_maze_area[s_index], mark))
					h_maze_area[s_index] = 0;
				if(TheSameNet(v_maze_area[s_index], mark))
					v_maze_area[s_index] = 0;
			}
			if(((s_maze_area_x-1 >= 0))&&
			  (s_maze_area_x+1 < maze_width))
			{
				s_index = (s_maze_area_y-1) * maze_width + s_maze_area_x+1;
				if(TheSameNet(h_maze_area[s_index], mark))
					h_maze_area[s_index] = 0;
				if(TheSameNet(v_maze_area[s_index], mark))
					v_maze_area[s_index] = 0;
			}
			x_diff2 = t_x - min_x_index;
			y_diff2 = t_y - min_y_index;
			maze_area_x_in_detailgrid = ((target.ax - CD->dmin_x) % levels[0].tile_width)/CD->wire_pitch;
			maze_area_y_in_detailgrid = ((target.ay - CD->dmin_y) % levels[0].tile_height)/CD->wire_pitch;
			
			int t_maze_area_x = dt_m*x_diff2 + maze_area_x_in_detailgrid;
			int t_maze_area_y = dt_n*y_diff2 + maze_area_y_in_detailgrid;

			int t_index = t_maze_area_y * maze_width + t_maze_area_x;
			h_maze_area[t_index] = v_maze_area[t_index] = CD->NetPin;//mark;

			int fail_pos;
			if(!DetailedMazeSearching(s_maze_area_x, s_maze_area_y, t_maze_area_x, t_maze_area_y, mark, CD->PinObstacle, fail_pos))
			{
				if(last_try)
				{
					int x_diff3 = (fail_pos % maze_width)/dt_m;
					int y_diff3 = (fail_pos / maze_width)/dt_n;
					int tile_index = left_bottom_corner_index + y_diff3*levels[0].n + x_diff3;
	
					DetailTile *dt = levels[0].tiles[tile_index].dt;
					dt->overflow++;
					
					edges_to_detail_route.erase(edges_to_detail_route.first());
					return false;
				}
				continue;
			}
			else
				BacktraceToFindPath(s_maze_area_x, s_maze_area_y, t_maze_area_x, t_maze_area_y, mark, e->route_type, e->vias);

			e->dt_route_grids.clear();
			//*Mark back to the corresponding grids in detail grids.
			//But source and target still be CD->PinObstacle

			for(int k=0;k<maze_width*maze_height;k++)
			{
				//find which detail tile is the maze area grids[k] in
				//x_diff3 is x difference between left_bottom_corner_index and j's detail tile.
				int x_diff3 = (k % maze_width)/dt_m;
				int y_diff3 = (k / maze_width)/dt_n;
				int tile_index = left_bottom_corner_index + y_diff3*levels[0].n + x_diff3;

				DetailTile *dt = levels[0].tiles[tile_index].dt;
				int maze_area_x_add = dt_m*x_diff3 + (maze_width-dt_m)*((k / maze_width)%dt_n);
				int maze_area_y_add = maze_width*dt_n*y_diff3;
				int maze_area_diff = maze_area_x_add+maze_area_y_add;
				int grid_index_in_dt = k - maze_area_diff;
				//mark means the grid is choosed to be routed.
				if(h_maze_area[k] == mark || h_maze_area[k] == CD->NetPin)
				{
					int *p;
					if(start_layer_type == HV_V)
						p = &dt->layers[where_is_start_layer+this_layer_with_up_layer_or_down].grids[grid_index_in_dt];
					else
						p = &dt->layers[where_is_start_layer].grids[grid_index_in_dt];
					*p = mark;
					dt->grids_used++;
					e->dt_grids++;
					if(multilevel_type == PerformanceDriven)
						e->dt_route_grids.push_back(p);
				}
				if(v_maze_area[k] == mark || v_maze_area[k] == CD->NetPin)
				{
					int *p;
					if(start_layer_type == HV_V)
						p = &dt->layers[where_is_start_layer].grids[grid_index_in_dt];
					else
						p = &dt->layers[where_is_start_layer+this_layer_with_up_layer_or_down].grids[grid_index_in_dt];
					*p = mark;
					dt->grids_used++;
					e->dt_grids++;
					if(multilevel_type == PerformanceDriven)
						e->dt_route_grids.push_back(p);

				}
			}
			e->dt_grids -= e->vias;

			//Mark pins unoccupied of this edge
			int g_x = ((source.ax - CD->dmin_x) % levels[0].tile_width)/CD->wire_pitch;
			int g_y = ((source.ay - CD->dmin_y) % levels[0].tile_height)/CD->wire_pitch;
			int index = g_y * dt_m + g_x;

			int max_layer = where_is_start_layer;
			if(this_layer_with_up_layer_or_down>0)
				max_layer += this_layer_with_up_layer_or_down;
			for(int k=0;k<CD->number_of_layers;k++)
			{
				DetailTile *dt = levels[0].tiles[Level0TileIndex(source)].dt;
				if(k>max_layer)
				{
					if(dt->layers[k].grids[index] == CD->PinObstacle)
						dt->layers[k].grids[index] = 0;
				}
			}
			g_x = ((target.ax - CD->dmin_x) % levels[0].tile_width)/CD->wire_pitch;
			g_y = ((target.ay - CD->dmin_y) % levels[0].tile_height)/CD->wire_pitch;
			index = g_y * dt_m + g_x;
			for(int k=0;k<CD->number_of_layers;k++)
			{
				DetailTile *dt = levels[0].tiles[Level0TileIndex(target)].dt;
				if(k>max_layer)
				{
					if(dt->layers[k].grids[index] == CD->PinObstacle)
						dt->layers[k].grids[index] = 0;
				}
			}
			break;
		}
		e->routed = true;
		e->route_type = Maze;
		edges_to_detail_route.erase(edges_to_detail_route.first());
	}
	return true;
}
/*
	Mark all pins in the grids of the corresponding detail tails occupied.
	This will suggest detail router avoid to use the detail grids of corresponding pins.
*/
void Multilevel::MarkAllPinsOccupied()
{
	for(int i=0;i<CD->cellinstancepins.size();i++)
	{
		CIP_ c = CD->cellinstancepins[i];
		int t_index = Level0TileIndex(c);
		
		int t_maze_area_x = ((c.ax - CD->dmin_x) % levels[0].tile_width)/CD->wire_pitch;
		int t_maze_area_y = ((c.ay - CD->dmin_y) % levels[0].tile_height)/CD->wire_pitch;
		int g_index = t_maze_area_y * dt_m + t_maze_area_x;

		DetailTile *dt = levels[0].tiles[t_index].dt;

		for(int j=0;j<CD->number_of_layers;j++)
		{
			dt->layers[j].grids[g_index] = CD->PinObstacle;
			dt->layers[j].grids[g_index] = CD->PinObstacle;
		}
	}
}
void Multilevel::DoMultilevel(int tiles)
{
	ConstructFirstLevel(sqrt(tiles));
	MarkAllPinsOccupied();
	Coarsen(tiles);
	UnCoarsen();
	for(int i=0;i<CD->nets.size();i++)
		RoughAnalysisNetTreeDelay(i);
	ReportEachNetTiming();
}

/*
	Bzedges_failed_to_detail_route_at_coarsen
*/
void Multilevel::PerformanceUnCoarsen()
{
	
	Edge_ *e;
	net_fail = 0;
	//refine edges_failed_to_detail_route_at_coarsen
	while(edges_failed_to_detail_route_at_coarsen.size())
	{
		list_item traveler = edges_failed_to_detail_route_at_coarsen.last();
		e = edges_failed_to_detail_route_at_coarsen[traveler];
		if(SatisfyRestrictedTimingConstraint(e->target))
		{
			while(1)
			{
				GlobalMazeRouting(e);
				if(e->routed)
				{
					if(SatisfyRestrictedTimingConstraint(e->target))	break;
					else
					{
						//clear dt_route_grids(reroute)
						for(int k=0;k<e->dt_route_grids.size();k++)
							*e->dt_route_grids[k] = 0;
						if(!RecallingModified(e))
						{
							e->route_type = NerverMeetTiming;
							break;
						}
					}
				}else
					break;
			}
		}else if(RecallingModified(e))
			continue;
		else
		{
			e->route_type = NerverMeetTiming;
		}
		edges_failed_to_detail_route_at_coarsen.erase(traveler);
	}
}

void Multilevel::ReportEachNetTiming()
{

	double max_delay = 9999;
	double total_max_delay = 0;
	double total_delay = 0;
	double total_n = 0;
	for(int i=0;i<CD->nets.size();i++)
	{
		Net_ *n = &CD->nets[i];
		bool first = true;
		n->delay_upper_bound = 0;
		for(int j=0;j<n->pins_uid.size();j++)
		{
			if(n->pins_uid[j] == n->source)	continue;

			double delay = CD->cellinstancepins[n->pins_uid[j]].delay;
			total_delay+=delay;
			total_n++;
			if(first)
			{
				n->delay_upper_bound = delay;
				first = false;
			}
			else if(n->delay_upper_bound < delay)
				n->delay_upper_bound = delay;
		}
		if(max_delay < n->delay_upper_bound)
			max_delay = n->delay_upper_bound;
		total_max_delay += n->delay_upper_bound;
		cout<<"Net "<<i<<" max delay:"<<n->delay_upper_bound<<" pins:"<<n->pins_uid.size()<<endl;
	}
	
	cout<<"#### max_delay ##### "<<max_delay<<endl;
	cout<<"#### avg_max_delay = "<<total_max_delay<<"/"<< CD->nets.size()<<"="<<total_max_delay/(double)CD->nets.size()<<"##### "<<endl;
	cout<<"#### avg_delay = "<<total_delay<<"/"<< total_n<<"="<<total_delay/(double)total_n<<"##### "<<endl;
}

bool Multilevel::NetsTimingAnalysis(int handle_never_meet)
{
int cc=0;
	cout<<"\n############## NetsTimingAnalysis Route Fail #####################\n";
	list_item t = net_to_be_handle.first();
	list_item pre_t;
	while(1)
	{
		Net_ *n = net_to_be_handle[t];
		int all_pass = true;
		for(int j=0;j<n->pins_uid.size();j++)
		{
			if(n->pins_uid[j] == n->source)	continue;

			CIP_ *cip = &CD->cellinstancepins[n->pins_uid[j]];
			Edge_ *e = CD->GetEdgeIn(n->pins_uid[j]);

			if(!handle_never_meet)
			{
				if(e->route_type == NerverMeetTiming)
				{
					all_pass = false;
					continue;
				}
			}
			if(!SatisfyRestrictedTimingConstraint(e->target))
			{
				all_pass = false;
				if(e->routed)
				{
					//clear dt_route_grids(reroute)
					for(int k=0;k<e->dt_route_grids.size();k++)
						*e->dt_route_grids[k] = 0;
				}
				if(!RecallingModified(e))
				{
					e->route_type = NerverMeetTiming;
					continue;
				}
				//refine stage like in PerformanceCoarsen stage
				while(1)
				{
					PatternRouting(e);
					edges_to_detail_route.push_back(e);
					if(!DetailRoute(1))
						ZShapeRouting(e);
					break;
				}
			}
		}
		pre_t = t;
		t = net_to_be_handle.succ(t);
		if(all_pass)
		{
			net_to_be_handle.erase(pre_t);
			//O
			for(int j=0;j<n->pins_uid.size();j++)
			{
				if(n->pins_uid[j] == n->source)	continue;
	
				CIP_ *cip = &CD->cellinstancepins[n->pins_uid[j]];
				Edge_ *e = CD->GetEdgeIn(n->pins_uid[j]);
				e->tile_routing.clear();
				e->dt_route_grids.clear();
			}
		}
		if(t==net_to_be_handle.last())
			break;
	}
}

void Multilevel::DoPerformanceMultilevel(int tiles)
{
	
	multilevel_type = Routibility;
	CD->ShortestPathRouting();
	
	DoMultilevel(tiles);
	constraint_net = 0;
	constraint_edges = 0;
	for(int i=0;i<CD->nets.size();i++)
		RoughAnalysisNetTreeDelay(i);

	CalculateNetUpperAndLowerBound();

	levels[0].hcapacity = levels[0].hcapacity = 0;
	for(int i=0;i<levels[0].tiles.size();i++)
	{
		DetailTile *dt = levels[0].tiles[i].dt;
		for(int j=0;j<CD->number_of_layers;j++)
			for(int k=0;k<dt->layers[0].grids.size();k++)
				dt->layers[j].grids[k] = 0;
	}

	multilevel_type = PerformanceDriven;
	MarkAllPinsOccupied();
	control_ration = 0.95;

	PerformanceCoarsen(tiles);
	control_ration = 0.97;

	net_to_be_handle.clear();
	for(int i=0;i<CD->nets.size();i++)
		net_to_be_handle.push_back(&CD->nets[i]);
	
	while(net_to_be_handle.size() && (control_ration<=1))
	{
		list_item t = net_to_be_handle.first();
		while(t)
		{
			RoughAnalysisNetTreeDelay(net_to_be_handle[t]->uid);
			t = net_to_be_handle.succ(t);
		}
		NetsTimingAnalysis(0);
		PerformanceUnCoarsen();
		control_ration+=0.05;
	}

	//last try, give those never_meet_edge last chance
	control_ration = 1;
	list_item t = net_to_be_handle.first();
	while(t)
	{
		RoughAnalysisNetTreeDelay(net_to_be_handle[t]->uid);
		t = net_to_be_handle.succ(t);
	}
	NetsTimingAnalysis(1);
	PerformanceUnCoarsen();

	for(int i=0;i<CD->nets.size();i++)
		RoughAnalysisNetTreeDelay(i);
	ReportEachNetTiming();
}