#include <LEDA/list.h>
#include <LEDA/ugraph.h> 
#include <LEDA/graph_alg.h>
#include <stdlib.h>
#include "customdesign.h"

CustomDesign::CustomDesign(char *filename)
{
	ldb.read (filename); //// Reads in a GDIF file
	Init();
}

double CustomDesign::DownStreamCapaciteance(int net_id, int cpi_id)
{
	Net_ *n = &nets[net_id];
	//CalculateDownStreamCapacitance, initial = this sink gate resistance
	double total_down_c = 0;
	
	leda_list <leda_edge> to_deal_edges = G.out_edges(cellinstancepins[cpi_id].n);
	while(1)
	{
		list_item t = to_deal_edges.first();
		total_down_c += MinGateCapacitance;
		if(!to_deal_edges.size())
			break;
		Edge_ *e = &edges[G.inf(to_deal_edges[t])];
		if(e->routed)
			total_down_c += (double)WireCapacitance * (double)e->dt_grids * (double)wire_pitch / (double)scale_factor;
		else
			total_down_c += (double)WireCapacitance * (double)e->distance * (double)wire_pitch / (double)scale_factor;
		leda_list <leda_edge> more_to_deal_edges = G.out_edges(cellinstancepins[e->target->uid].n);
		to_deal_edges.conc(more_to_deal_edges, LEDA::after);
		to_deal_edges.erase(t);
	}
	return total_down_c;
}

/*
	input : a net, a sink
	use net topology to calculate Delay
	return Elmore delay of a sink from source (psec)
*/
double CustomDesign::NetElmoreDelay(int net_id, int cpi_id)//, int &vias)
{
	Net_ *n = &nets[net_id];
	//CalculateDownStreamCapacitance, initial = this sink gate resistance
	double total_down_c = 0;
	double total_delay = 0;
	
	leda_list <leda_edge> to_deal_edges = G.in_edges(cellinstancepins[cpi_id].n);
	while(1)
	{
		double resistance = 0;
		double capacitance = 0;
		double this_wire_capacitance = 0;
		double this_wire_resistance = 0;
		double down_stream_capaciteance = 0;
		list_item t = to_deal_edges.first();
		if(!to_deal_edges.size())
			break;
		Edge_ *e = &edges[G.inf(to_deal_edges[t])];
		down_stream_capaciteance = DownStreamCapaciteance(net_id, e->target->uid);
		//more accurate calculation
		if(e->routed)
		{
			resistance = (double)WireResistance * (double)e->dt_grids * (double)wire_pitch / (double)scale_factor;
			this_wire_capacitance = ((double)WireCapacitance * (double)e->dt_grids * (double)wire_pitch / (double)scale_factor);
			//consider gate
			total_delay += MinGateResistance * (this_wire_capacitance + down_stream_capaciteance);
			//consider wire
			total_delay += this_wire_capacitance * (this_wire_capacitance/2 + down_stream_capaciteance);
			total_delay += (double)e->vias * ViaDelay;
		}
		else
		{
			resistance = (double)WireResistance * (double)e->distance * (double)wire_pitch / (double)scale_factor;
			this_wire_capacitance = ((double)WireCapacitance * (double)e->distance * (double)wire_pitch / (double)scale_factor);

			//consider gate
			total_delay += MinGateResistance * (this_wire_capacitance + down_stream_capaciteance);
			//consider wire
			total_delay += this_wire_capacitance * (this_wire_capacitance/2 + down_stream_capaciteance);
			total_delay += 1 * ViaDelay;
		}

		leda_list <leda_edge> more_to_deal_edges = G.in_edges(cellinstancepins[e->source->uid].n);
		to_deal_edges.conc(more_to_deal_edges, LEDA::after);
		to_deal_edges.erase(t);
	}
	//now we have the downstream of cpi_id, calculate delay from source
	cellinstancepins[cpi_id].delay = total_delay;
	return total_delay;
	
}

/*
	input : a edge of graph G
	return twp CIP_ representing the source of edge and traget of it.
*/
void CustomDesign::GetEdgeST(leda_edge e, CIP_ &source, CIP_ &target)
{
	source = cellinstancepins[G.inf(G.source(e))];
	target = cellinstancepins[G.inf(G.target(e))];
}
/*
	input : a cellinstancepin
	return the Edge_ which goes in to the pin
*/
Edge_* CustomDesign::GetEdgeIn(int cip_index)
{
	CIP_ *cip = &cellinstancepins[cip_index];
	leda_list <leda_edge> e = G.in_edges(cip->n);
	if(!e.size()){cout<<"Error\n";exit(0);}
	return &edges[G.inf(e[e.first()])];
}
/*
	Get CI_ of cell instance with name
*/
CI_* CustomDesign::GetCIByName(const char *name)
{
	return &cellinstances[Cimap[name]];
}

/*
	Get the CIP_ of the pin with pin_name and cell_name
*/
CIP_* CustomDesign::GetCIPByName(const char *cell_name, const char *pin_name)
{
	CI_ *ci = GetCIByName(cell_name);
	for(int i=0;i<ci->pins_uid.size();i++)
	{
		if(!strcmp(cellinstancepins[ci->pins_uid[i]].name, pin_name))
			return &cellinstancepins[ci->pins_uid[i]];
	}
	return NULL;
}
int CustomDesign::ManhattanDistance(CIP_ cip1, CIP_ cip2)
{
	int c1_x = (cip1.ax-dmin_x-1) / wire_pitch + 1;
	int c1_y = (cip1.ay-dmin_y-1) / wire_pitch + 1;
	int c2_x = (cip2.ax-dmin_x-1) / wire_pitch + 1;
	int c2_y = (cip2.ay-dmin_y-1) / wire_pitch + 1;
	return (abs(c1_x-c2_x)+abs(c1_y-c2_y));
}

PinType CustomDesign::GetPinType(const char* name)
{
	if(!strcmp(name, ""))	return UNKNOWN;
	int index = 0;
	while(name[index+1]!='\0')
	{
		switch(name[index])
		{
			case 'P':
				if(name[index+1] == 'I')	return IN;
				else if(name[index+1] == 'O')	return OUT;
				break;
			case 'O':
				if(name[index] == 'I' && name[index+1] == 'N')	return IN;
				else if(name[index+2] != '\0')
				{
					if(name[index] == 'O' && name[index+1] == 'U' &&
					name[index+2] == 'T')	return OUT;
				}
				break;
			default:break;
		}
		index++;
	}
	return UNKNOWN;
}

/*
	ShortestPathRouting is used to calculate the low bound for each output pins.
*/
void CustomDesign::ShortestPathRouting()
{
	G.del_all_edges();
	edges.clear();
	for(int i=0;i<nets.size();i++)
	{
		for(int j=0;j<nets[i].pins_uid.size();j++)
		{
			if(nets[i].source != nets[i].pins_uid[j])
				AssignEdge(nets[i].source, nets[i].pins_uid[j]);
		}
	}
	PinObstacle = (-1)*edges.size()-1;
	NetPin = PinObstacle-1;
}
/*
	Initialize our data structure by information of LayoutDatabase 
	(structure is used to store additional variables).
*/
void CustomDesign::Init()
{
	cellinstancepins.clear();
	cellinstances.clear();
	Cimap.clear();
	Lid dbId = ldb.id (); //// gets the Lid of the database
	//record Parameters
	vector<int> v;
	v = ldb.getIntParams ("number_of_layers");
	number_of_layers = (v[0] / 1000);
	v = ldb.getIntParams ("wire_spacings");
	wire_spacings = v[0];
	v = ldb.getIntParams ("wire_widths");
	wire_widths = v[0];
	wire_pitch = wire_spacings + wire_widths;
	//end of recording Parameters
	for (Lid cell = dbId.first (Lid::Cell); cell; ++cell)
	{
		if(strcmp(cell.name (),"top")&&strcmp(cell.name (),"design"))
			continue;
		CellInstance_ celli;
		celli.uid = cellinstances.size();
		celli.pins_uid.clear();
		celli.name = cell.name ();
		for (Lid port = cell.first(Lid::Port); port; ++port)
		{
			CellInstancePin_ pin;
			pin.name = port.name ();
			pin.ax = port.x();
			pin.ay = port.y();
			if(pin.ax < dmin_x)	dmin_x = pin.ax;
			if(pin.ax > dmax_x)	dmax_x = pin.ax;
			if(pin.ay < dmin_y)	dmin_y = pin.ay;
			if(pin.ay > dmax_y)	dmax_y = pin.ay;
			pin.uid = cellinstancepins.size();
			pin.ci_uid = cellinstances.size();
			pin.type = GetPinType(port.name ());
			if(pin.type == OUT)
				celli.outpin = cellinstancepins.size();
			leda_node n;
			n = G.new_node();
			G.assign(n, cellinstancepins.size());
			pin.n = n;
			celli.pins_uid.push_back(cellinstancepins.size());
			cellinstancepins.push_back(pin);
		}
		Cimap.insert(make_pair(string(cell.name ()), cellinstances.size()));
		celli.not_conn_pins = celli.pins_uid.size();
		cellinstances.push_back(celli);

		//build cellinstance and port information
		for (Lid inst = cell.first(Lid::CellInstance); inst; ++inst)
		{
			CellInstance_ celli;
			celli.uid = cellinstances.size();
			celli.pins_uid.clear();
			celli.name = inst.name ();
			//// Get the first port from the cell and iterates over all the ports
			Lid target_cell = dbId.first (Lid::Cell, inst.cellRef().name ());
			for (Lid port = target_cell.first(Lid::Port); port; ++port)
			{
				CellInstancePin_ pin;
				pin.name = port.name ();
				if(inst.orient() == 0)
				{
					pin.ax = inst.x() + port.x ();
					pin.ay = inst.y() + port.y ();
				}else if(inst.orient() == 90000)
				{
					pin.ax = inst.x() + port.y ();
					pin.ay = inst.y() - port.x ();
				}else if(inst.orient() == 180000)
				{
					pin.ax = inst.x() - port.x ();
					pin.ay = inst.y() - port.y ();
				}else if(inst.orient() == 270000)
				{
					pin.ax = inst.x() - port.y ();
					pin.ay = inst.y() + port.x ();
				}

				if(pin.ax < dmin_x)	dmin_x = pin.ax;
				if(pin.ax > dmax_x)	dmax_x = pin.ax;
				if(pin.ay < dmin_y)	dmin_y = pin.ay;
				if(pin.ay > dmax_y)	dmax_y = pin.ay;

				pin.uid = cellinstancepins.size();
				pin.ci_uid = cellinstances.size();
				pin.type = GetPinType(port.name ());
				if(pin.type == OUT)
					celli.outpin = cellinstancepins.size();
				leda_node n;
				n = G.new_node();
				G.assign(n, cellinstancepins.size());
				pin.n = n;
				celli.pins_uid.push_back(cellinstancepins.size());
				cellinstancepins.push_back(pin);
			}
			Cimap.insert(make_pair(string(inst.name ()), cellinstances.size()));
			celli.not_conn_pins = celli.pins_uid.size();
			cellinstances.push_back(celli);
		}

		//build net information of all ports;
		for (Lid net = cell.first (Lid::Net); net; ++net)
		{
			Net_ n;
			n.name = net.name();
			n.pins_uid.clear();
			n.uid = nets.size();
			n.source = -100;
			CIP_ *pin;
			char pre_port[30],pre_inst[30];
			char now_port[30],now_inst[30];
			for (Lid ref = net.first (Lid::NetRef); ref; ++ref)
			{
				if (ref.type () == Lid::InstPortRef)
				{
					strcpy(now_port, ref.portRef().name());
					strcpy(now_inst, ref.instRef().name());
					//mcc1_1d_gp.gdf 5371榳error (portRef P438 (instRef U1201))
//					cout<<"pre_port:"<<pre_port<<", now_port:"<<now_port<<endl;
					if (!strcmp(pre_port, now_port))
					{
//						cout<<"pre_inst:"<<pre_inst<<", now_inst:"<<now_inst<<endl;
						if (!strcmp(pre_inst, now_inst))
							continue;
//						{cout<<"wa le\n\n";exit(0);}
					}
					pin = GetCIPByName(ref.instRef().name(), ref.portRef().name());
					//make net information
					pin->net_uid = nets.size();
					strcpy(pre_port, ref.portRef().name());
					strcpy(pre_inst, ref.instRef().name());
				}else if(ref.type () == Lid::PortRef)
				{
					pin = GetCIPByName(cell.name(), ref.portRef().name());
					//make net information
					pin->net_uid = nets.size();
				}
				else
				{
					cout <<"Error!Port type \"pathref\" not handeled!!\n";
					exit(0);
				}
				n.pins_uid.push_back(pin->uid);
				if((pin->type == OUT && ref.type () != Lid::PortRef) ||
				    (ref.type () == Lid::PortRef && pin->type == IN))
					n.source = pin->uid;
			}
			if(n.source == -100)
				n.source = pin->uid;
			nets.push_back(n);
		}
	}
}

/*
	Assign a edge to two cell instance pins
*/
void CustomDesign::UpdateEdge(int edges_index, int source, int target)
{
	Edge_ *ee = &edges[edges_index];
	CIP_ *cip1 = &cellinstancepins[source];
	CIP_ *cip2 = &cellinstancepins[target];

	if(ee->e)
		G.del_edge(ee->e);

	leda_edge e;
	e = G.new_edge(cip1->n, cip2->n);
	G.assign(e, edges_index);
	
	int delay = ManhattanDistance(*cip1, *cip2);
	ee->source = cip1;
	ee->target = cip2;
	ee->e = e;
	ee->level = -1;
	ee->route_type = Unrouted;	//Edge is now unrouted
	ee->vias = 0;
	ee->dt_grids = 0;
	ee->routed = false;
	ee->distance = delay;
}
/*
	Assign a edge to two cell instance pins
*/
void CustomDesign::AssignEdge(int source, int target)
{
	Edge_ ee;
	CIP_ *cip1 = &cellinstancepins[source];
	CIP_ *cip2 = &cellinstancepins[target];
	leda_edge e;
	e = G.new_edge(cip1->n, cip2->n);
	
	int delay = ManhattanDistance(*cip1, *cip2);
	G.assign(e, edges.size());
	ee.source = cip1;
	ee.target = cip2;
	ee.e = e;
	ee.uid = edges.size();
	ee.level = -1;
	ee.route_type = Unrouted;	//Edge is now unrouted
	ee.vias = 0;
	ee.dt_grids = 0;
	ee.routed = false;
	ee.distance = delay;
	edges.push_back(ee);
}

void CustomDesign::MinimumSpanningTreeRouting()
{
	G.del_all_edges();
	edges.clear();
	leda_edge e;

int max_pins = 0;
int l_100 = 0;
int l_200 = 0;
int l_300 = 0;
int l_400 = 0;
	for(int i=0;i<nets.size();i++)
	{
		if(max_pins<nets[i].pins_uid.size()){max_pins=nets[i].pins_uid.size();}
		if(100>nets[i].pins_uid.size()) l_100++;
		else if(200>nets[i].pins_uid.size()) l_200++;
		else if(300>nets[i].pins_uid.size()) l_300++;
		else l_400++;
		Net_ *n = &nets[i];
		GRAPH <int, int> MSTG;
		GRAPH <int, int> BFSG;
		vector <leda_node> nodes;
		vector <leda_node> BFSnodes;
		nodes.clear();
		BFSnodes.clear();
		
		leda_node source_node;
		MSTG.clear();
		BFSG.clear();
		//each sink of this net assigns a node to MSTG
		for(int j=0;j<nets[i].pins_uid.size();j++)
		{
			int cip_index = nets[i].pins_uid[j];
			CIP_ *cip = &cellinstancepins[cip_index];
			leda_node nn, before_nn;
			nn = MSTG.new_node();
			MSTG.assign(nn, j);
			nodes.push_back(nn);
			nn = BFSG.new_node();
			BFSnodes.push_back(nn);
			if(nets[i].pins_uid[j] == nets[i].source)
				source_node = nn;
		}
		leda_edge e1;
		//every two nodes consist a bidirected edge
		for(int j=0;j<nodes.size();j++)
		{
			leda_node n1 = nodes[j];
			for(int k=j+1;k<nodes.size();k++)
			{
				leda_node n2 = nodes[k];
				int e_cost = ManhattanDistance(cellinstancepins[nets[i].pins_uid[MSTG.inf(n1)]], cellinstancepins[nets[i].pins_uid[MSTG.inf(n2)]]);
				e1 = MSTG.new_edge(n1, n2);
				MSTG.assign(e1, e_cost);
				e1 = MSTG.new_edge(n2, n1);
				MSTG.assign(e1, e_cost);
			}
		}

		leda_edge_array<int> cost(MSTG);
		forall_edges(e1,MSTG)
			cost[e1] = MSTG.inf(e1);

		leda_list<leda_edge> net_edges = MIN_SPANNING_TREE(MSTG, cost);
		list_item t = net_edges.first();

		//every two nodes consist a bidirected edge
		while(t)
		{
			e1 = net_edges[t];;
			int BFS_node_index1 = MSTG.inf(MSTG.source(e1));
			int BFS_node_index2 = MSTG.inf(MSTG.target(e1));
			leda_edge e2 = BFSG.new_edge(BFSnodes[BFS_node_index1], BFSnodes[BFS_node_index2]);
			e2 = BFSG.new_edge(BFSnodes[BFS_node_index2], BFSnodes[BFS_node_index1]);
			t = net_edges.succ(t);
		}

		//from source_node to build our spanning tree
		leda_node_array<int> dist(BFSG, -1);

		BFS(BFSG, source_node, dist);

		t = net_edges.first();
		while(t)
		{
			e1 = net_edges[t];;
			int source = dist[BFSnodes[MSTG.inf(MSTG.source(e1))]];
			int target = dist[BFSnodes[MSTG.inf(MSTG.target(e1))]];
			int pin1 = nets[i].pins_uid[MSTG.inf(MSTG.source(e1))];
			int pin2 = nets[i].pins_uid[MSTG.inf(MSTG.target(e1))];
			if(source < target)
				AssignEdge(pin1, pin2);
			else
				AssignEdge(pin2, pin1);
			t = net_edges.succ(t);
		}
	}
cout<<"file information:\n";
cout<<"Max pins of a net:"<<max_pins<<endl;
cout<<"The number of nets contain less than 100 pins:"<<l_100<<endl;
cout<<"The number of nets contain less than 200 pins:"<<l_200<<endl;
cout<<"The number of nets contain less than 300 pins:"<<l_300<<endl;
cout<<"The number of nets contain more than 300 pins:"<<l_400<<endl;
cout<<"Number of nets:"<<nets.size()<<endl;
cout<<"Number of nets after MST decomposition:"<<edges.size()<<endl;
	PinObstacle = (-1)*edges.size()-1;
	NetPin = PinObstacle-1;
}

